<?php
// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

/**
 * After setup theme hook
 */
function floral_fashion_theme_setup(){

    /*
     * Make child theme available for translation.
     * Translations can be filed in the /languages/ directory.
     */
     load_child_theme_textdomain( 'floral-fashion', get_stylesheet_directory() . '/languages' );

     add_image_size( 'blossom-floral-blog-list-small', 345, 195, true );
}
add_action( 'after_setup_theme', 'floral_fashion_theme_setup', 100 );

/**
 * Load assets.
 */
function floral_fashion_enqueue_styles() {

    $my_theme = wp_get_theme();
    $version = $my_theme['Version'];
    
    wp_enqueue_style( 'blossom-floral', get_template_directory_uri() . '/style.css' );
    wp_enqueue_style( 'floral-fashion', get_stylesheet_directory_uri() . '/style.css', array( 'blossom-floral' ), $version );

    wp_enqueue_script( 'floral-fashion', get_stylesheet_directory_uri() . '/js/custom.js', array( 'jquery'), $version, true );
    wp_enqueue_script( 'floral-fashion-acessibility', get_stylesheet_directory_uri() . '/js/modal-accessibility.js', array( 'jquery' ), $version, true );

}
add_action( 'wp_enqueue_scripts', 'floral_fashion_enqueue_styles', 10 );

function floral_fashion_customize_script(){

    $my_theme = wp_get_theme();
    $version = $my_theme['Version'];
    wp_enqueue_script( 'floral-fashion-customize', get_stylesheet_directory_uri() . '/js/customize.js', array( 'jquery', 'customize-controls' ), $version, true );

}
add_action( 'customize_controls_enqueue_scripts', 'floral_fashion_customize_script' );



//Remove a function from the parent theme
function floral_fashion_remove_parent_filters(){ //Have to do it after theme setup, because child theme functions are loaded first
    remove_action( 'customize_register', 'blossom_floral_customize_register_appearance' );
    remove_action( 'wp_head', 'blossom_floral_dynamic_css', 99 );
}
add_action( 'init', 'floral_fashion_remove_parent_filters' );


function floral_fashion_customize_register( $wp_customize ) {
    $wp_customize->add_section( 'theme_info', array(
        'title'       => __( 'Demo & Documentation' , 'floral-fashion' ),
        'priority'    => 6,
    ) );
    
    /** Important Links */
    $wp_customize->add_setting( 'theme_info_theme',
        array(
            'default' => '',
            'sanitize_callback' => 'wp_kses_post',
        )
    );
    
    $theme_info = '<p>';
    $theme_info .= sprintf( __( 'Demo Link: %1$sClick here.%2$s', 'floral-fashion' ),  '<a href="' . esc_url( 'https://blossomthemes.com/theme-demo/?theme=floral-fashion' ) . '" target="_blank">', '</a>' ); 
    $theme_info .= '</p><p>';
    $theme_info .= sprintf( __( 'Documentation Link: %1$sClick here.%2$s', 'floral-fashion' ),  '<a href="' . esc_url( 'https://docs.blossomthemes.com/floral-fashion/' ) . '" target="_blank">', '</a>' ); 
    $theme_info .= '</p>';

    $wp_customize->add_control( new Blossom_Floral_Note_Control( $wp_customize,
        'theme_info_theme', 
            array(
                'section'     => 'theme_info',
                'description' => $theme_info
            )
        )
    );

    /** Typography Settings */
    $wp_customize->add_panel(
        'appearance_settings',
        array(
            'title'       => __( 'Appearance Settings', 'floral-fashion' ),
            'capability'  => 'edit_theme_options',
            'priority'    => 30,
        )
    );

    /** Primary Color*/
    $wp_customize->add_setting( 
        'primary_color', 
        array(
            'default'           => '#fdbea9',
            'sanitize_callback' => 'sanitize_hex_color',
        ) 
    );

    $wp_customize->add_control( 
        new WP_Customize_Color_Control( 
            $wp_customize, 
            'primary_color', 
            array(
                'label'       => __( 'Primary Color', 'floral-fashion' ),
                'description' => __( 'Primary color of the theme.', 'floral-fashion' ),
                'section'     => 'colors',
                'priority'    => 5,
            )
        )
    );

    
    /** Secondary Color*/
    $wp_customize->add_setting(
        'secondary_color',
        array(
            'default'           => '#ddbea9',
            'sanitize_callback' => 'sanitize_hex_color',
        )
    );

    $wp_customize->add_control(
        new WP_Customize_Color_Control(
            $wp_customize,
            'secondary_color',
            array(
                'label'       => __('Secondary Color', 'floral-fashion'),
                'description' => __('Secondary color of the theme.', 'floral-fashion'),
                'section'     => 'colors',
                'priority'    => 6,
            )
        )
    );

    
    /** Typography */
    $wp_customize->add_section(
        'typography_settings',
        array(
            'title'    => __( 'Typography', 'floral-fashion' ),
            'priority' => 15,
            'panel'    => 'appearance_settings',
        )
    );

    /** Primary Font */
	$wp_customize->add_setting(
		'primary_font',
		array(
			'default'			=> 'Inter',
			'sanitize_callback' => 'blossom_floral_sanitize_select'
		)
	);

	$wp_customize->add_control(
		new Blossom_Floral_Select_Control(
			$wp_customize,
			'primary_font',
			array(
				'label'	      => __('Primary Font', 'floral-fashion'),
				'description' => __('Primary font of the site.', 'floral-fashion'),
				'section'     => 'typography_settings',
				'choices'     => blossom_floral_get_all_fonts(),
			)
		)
	);

    /** Secondary Font */
	$wp_customize->add_setting(
		'secondary_font',
		array(
			'default'			=> 'Source Serif Pro',
			'sanitize_callback' => 'blossom_floral_sanitize_select'
		)
	);

	$wp_customize->add_control(
		new Blossom_Floral_Select_Control(
			$wp_customize,
			'secondary_font',
			array(
				'label'	      => __('Secondary Font', 'floral-fashion'),
				'description' => __('Secondary font of the site.', 'floral-fashion'),
				'section'     => 'typography_settings',
				'choices'     => blossom_floral_get_all_fonts(),
			)
		)
	);

    $wp_customize->add_setting(
        'ed_localgoogle_fonts',
        array(
            'default'           => false,
            'sanitize_callback' => 'blossom_floral_sanitize_checkbox',
        )
    );
    
    $wp_customize->add_control(
        new Blossom_floral_Toggle_Control( 
            $wp_customize,
            'ed_localgoogle_fonts',
            array(
                'section'       => 'typography_settings',
                'label'         => __( 'Load Google Fonts Locally', 'floral-fashion' ),
                'description'   => __( 'Enable to load google fonts from your own server instead from google\'s CDN. This solves privacy concerns with Google\'s CDN and their sometimes less-than-transparent policies.', 'floral-fashion' )
            )
        )
    );   

    $wp_customize->add_setting(
        'ed_preload_local_fonts',
        array(
            'default'           => false,
            'sanitize_callback' => 'blossom_floral_sanitize_checkbox',
        )
    );
    
    $wp_customize->add_control(
        new Blossom_floral_Toggle_Control( 
            $wp_customize,
            'ed_preload_local_fonts',
            array(
                'section'       => 'typography_settings',
                'label'         => __( 'Preload Local Fonts', 'floral-fashion' ),
                'description'   => __( 'Preloading Google fonts will speed up your website speed.', 'floral-fashion' ),
                'active_callback' => 'blossom_floral_ed_localgoogle_fonts'
            )
        )
    );   

    ob_start(); ?>
        
        <span style="margin-bottom: 5px;display: block;"><?php esc_html_e( 'Click the button to reset the local fonts cache', 'floral-fashion' ); ?></span>
        
        <input type="button" class="button button-primary blossom-floral-flush-local-fonts-button" name="blossom-floral-flush-local-fonts-button" value="<?php esc_attr_e( 'Flush Local Font Files', 'floral-fashion' ); ?>" />
    <?php
    $floral_fashion_flush_button = ob_get_clean();

    $wp_customize->add_setting(
        'ed_flush_local_fonts',
        array(
            'sanitize_callback' => 'wp_kses_post',
        )
    );
    
    $wp_customize->add_control(
        'ed_flush_local_fonts',
        array(
            'label'         => __( 'Flush Local Fonts Cache', 'floral-fashion' ),
            'section'       => 'typography_settings',
            'description'   => $floral_fashion_flush_button,
            'type'          => 'hidden',
            'active_callback' => 'blossom_floral_ed_localgoogle_fonts'
        )
    );
    
    /** Move Background Image section to appearance panel */
    $wp_customize->get_section('colors')->panel                          = 'appearance_settings';
    $wp_customize->get_section('colors')->priority                       = 10;
    $wp_customize->get_section('background_image')->panel                = 'appearance_settings';
    $wp_customize->get_section('background_image')->priority             = 15;

    $wp_customize->add_panel(
        'layout_settings',
        array(
            'title'    => __( 'Layout Settings', 'floral-fashion' ),
            'priority'    => 60,
            'capability'  => 'edit_theme_options',
        )
    );

    /** Header Layout Settings */
    $wp_customize->add_section(
        'header_layout_settings',
        array(
            'title'    => __( 'Header Layout', 'floral-fashion' ),
            'priority' => 10,
            'panel'    => 'layout_settings',
        )
    );

    /** Page Sidebar layout */
    $wp_customize->add_setting( 
        'header_layout', 
        array(
            'default'           => 'four',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );

    $wp_customize->add_control(
        new Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'header_layout',
            array(
                'section'	  => 'header_layout_settings',
                'label'		  => __( 'Header Layout', 'floral-fashion' ),
                'description' => __( 'Choose the layout of the header for your site.', 'floral-fashion' ),
                'choices'	  => array(
                    'one' => get_stylesheet_directory_uri() . '/images/header/one.svg',
                    'four' => get_stylesheet_directory_uri() . '/images/header/four.svg',
                )
            )
        )
    );

    $wp_customize->add_setting(
        'header_layout_text',
        array(
            'default'           => '',
            'sanitize_callback' => 'wp_kses_post' 
        )
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Note_Control( 
            $wp_customize,
            'header_layout_text',
            array(
                'section'     => 'header_layout_settings',
                'description' => sprintf( __( '%1$sClick here%2$s to change header settings.', 'floral-fashion' ), '<span class="text-inner-link header_layout_text">', '</span>' ),
            )
        )
    ); 

    $wp_customize->add_setting(
        'header_setting_text',
        array(
            'default'           => '',
            'sanitize_callback' => 'wp_kses_post' 
        )
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Note_Control( 
            $wp_customize,
            'header_setting_text',
            array(
                'section'     => 'header_settings',
                'description' => sprintf( __( '%1$sClick here%2$s to change header layout.', 'floral-fashion' ), '<span class="text-inner-link header_setting_text">', '</span>' ),
            )
        )
    );

    /** Slider Layout Settings */
    $wp_customize->add_section(
        'slider_layout_settings',
        array(
            'title'    => __( 'Slider Layout', 'floral-fashion' ),
            'priority' => 20,
            'panel'    => 'layout_settings',
        )
    );
    
    $wp_customize->add_setting( 
        'slider_layout', 
        array(
            'default'           => 'eight',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'slider_layout',
            array(
                'section'     => 'slider_layout_settings',
                'label'       => __( 'Slider Layout', 'floral-fashion' ),
                'description' => __( 'Choose the layout of the slider for your site.', 'floral-fashion' ),
                'choices'     => array(
                    'one'   => get_stylesheet_directory_uri() . '/images/slider/one.svg',
                    'eight'   => get_stylesheet_directory_uri() . '/images/slider/eight.svg',
                )
            )
        )
    );

    $wp_customize->add_setting(
        'slider_banner_layout_text',
        array(
            'default'           => '',
            'sanitize_callback' => 'wp_kses_post' 
        )
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Note_Control( 
            $wp_customize,
            'slider_banner_layout_text',
            array(
                'section'     => 'slider_layout_settings',
                'description' => sprintf( __( '%1$sClick here%2$s to configure slider settings.', 'floral-fashion' ), '<span class="text-inner-link slider_banner_layout_text">', '</span>' ),
            )
        )
    );

    $wp_customize->add_setting(
        'slider_banner_text',
        array(
            'default'           => '',
            'sanitize_callback' => 'wp_kses_post' 
        )
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Note_Control( 
            $wp_customize,
            'slider_banner_text',
            array(
                'section'     => 'header_image',
                'description' => sprintf( __( '%1$sClick here%2$s to select the layout of slider banner.', 'floral-fashion' ), '<span class="text-inner-link slider_banner_text">', '</span>' ),
            )
        )
    );
    
     /** Home Layout Settings */
     $wp_customize->add_section(
        'home_layout_settings',
        array(
            'title'    => __( 'Home Page Layout', 'floral-fashion' ),
            'priority' => 30,
            'panel'    => 'layout_settings',
        )
    );
    
    $wp_customize->add_setting( 
        'homepage_layout', 
        array(
            'default'           => 'two',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );
    
    $wp_customize->add_control(
        new  Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'homepage_layout',
            array(
                'section'     => 'home_layout_settings',
                'label'       => __( 'Homepage Layout', 'floral-fashion' ),
                'description' => __( 'Choose the layout of the homepage for your site.', 'floral-fashion' ),
                'choices'     => array(
                    'one'   => get_stylesheet_directory_uri() . '/images/home/one.svg',
                    'two'   => get_stylesheet_directory_uri() . '/images/home/two.svg',
                )
            )
        )
    );

    /** General Sidebar Layout Settings */
    $wp_customize->add_section(
        'general_layout_settings',
        array(
            'title'    => __( 'General Sidebar Layout', 'floral-fashion' ),
            'panel'    => 'layout_settings'
        )
    );
    
    /** Page Sidebar layout */
    $wp_customize->add_setting( 
        'page_sidebar_layout', 
        array(
            'default'           => 'right-sidebar',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'page_sidebar_layout',
            array(
                'section'     => 'general_layout_settings',
                'label'       => __( 'Page Sidebar Layout', 'floral-fashion' ),
                'description' => __( 'This is the general sidebar layout for pages. You can override the sidebar layout for individual page in respective page.', 'floral-fashion' ),
                'choices'     => array(
                    'no-sidebar'    => esc_url( get_template_directory_uri() . '/images/1c.jpg' ),
                    'centered'      => esc_url( get_template_directory_uri() . '/images/1cc.jpg' ),
                    'left-sidebar'  => esc_url( get_template_directory_uri() . '/images/2cl.jpg' ),
                    'right-sidebar' => esc_url( get_template_directory_uri() . '/images/2cr.jpg' ),
                )
            )
        )
    );

    /** Post Sidebar layout */
    $wp_customize->add_setting( 
        'post_sidebar_layout', 
        array(
            'default'           => 'right-sidebar',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );
    
    $wp_customize->add_control(
        new Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'post_sidebar_layout',
            array(
                'section'     => 'general_layout_settings',
                'label'       => __( 'Post Sidebar Layout', 'floral-fashion' ),
                'description' => __( 'This is the general sidebar layout for posts & custom post. You can override the sidebar layout for individual post in respective post.', 'floral-fashion' ),
                'choices'     => array(
                    'no-sidebar'    => esc_url( get_template_directory_uri() . '/images/1c.jpg' ),
                    'centered'      => esc_url( get_template_directory_uri() . '/images/1cc.jpg' ),
                    'left-sidebar'  => esc_url( get_template_directory_uri() . '/images/2cl.jpg' ),
                    'right-sidebar' => esc_url( get_template_directory_uri() . '/images/2cr.jpg' ),
                )
            )
        )
    );
    

        /** Default Sidebar layout */
    $wp_customize->add_setting(
        'layout_style', 
        array(
            'default'           => 'right-sidebar',
            'sanitize_callback' => 'blossom_floral_sanitize_radio'
        ) 
    );

    $wp_customize->add_control(
        new Blossom_Floral_Radio_Image_Control(
            $wp_customize,
            'layout_style',
            array(
                'section'     => 'general_layout_settings',
                'label'       => __( 'Default Sidebar Layout', 'floral-fashion' ),
                'description' => __( 'This is the general sidebar layout for whole site.', 'floral-fashion' ),
                'choices'     => array(
                    'no-sidebar'    => esc_url( get_template_directory_uri() . '/images/1c.jpg' ),
                    'left-sidebar'  => esc_url( get_template_directory_uri() . '/images/2cl.jpg' ),
                    'right-sidebar' => esc_url( get_template_directory_uri() . '/images/2cr.jpg' ),
                )
            )
        )
    );

}
add_action( 'customize_register', 'floral_fashion_customize_register', 90 );

function blossom_floral_body_classes( $classes ) {
    $layout = get_theme_mod( 'homepage_layout', 'two' );
    
    if( $layout == 'one') {
        $home_layout = 'list-layout'; 
    }else{
        $home_layout = 'grid-layout first';
    }

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}
    
    // Adds a class of custom-background-image to sites with a custom background image.
	if ( get_background_image() ) {
		$classes[] = 'custom-background-image';
	}
    
    // Adds a class of custom-background-color to sites with a custom background color.
    if ( get_background_color() != 'ffffff' ) {
		$classes[] = 'custom-background-color';
	}
    
    // Adds a class single post layout.
    if( is_single() ) {
        $classes[] = 'style-one';
    }

    if ( is_home() || ( is_archive() && !( blossom_floral_is_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() ) ) ) || is_search() ) {
        $classes[] = $home_layout;
    }

    $classes[] = blossom_floral_sidebar( true );
    
	return $classes;
}


function blossom_floral_header(){ 

    $ed_social      = get_theme_mod('ed_social_links', false);
    $ed_search      = get_theme_mod( 'ed_header_search', false );
    $ed_cart        = get_theme_mod( 'ed_shopping_cart', false );
    $header_bg_img2 = get_theme_mod( 'header_bg_image_two' );
    $add_class      = ! empty($header_bg_img2) ? " header-img" : '';
    $add_class_two  = ( !$ed_social && !$ed_search && !$ed_cart ) ? " site-branding-only" : '';
    $header_layout  = get_theme_mod( 'header_layout', 'four' ); ?>

    <header id="masthead" class="site-header style-<?php echo esc_attr( $header_layout ); echo esc_attr( $add_class ); ?>" itemscope itemtype="http://schema.org/WPHeader">
        <?php if( $header_layout == 'one') { ?>
            <div class="header-top">
                <div class="container">
                    <div class="header-right">
                        <?php blossom_floral_secondary_navigation(); ?>
                    </div>
                </div>
            </div>
            <div class="header-middle">
                <div class="container<?php echo esc_attr( $add_class_two ); ?>">
                    <?php if( $ed_social ) { ?>
                        <div class="header-left">
                            <?php blossom_floral_social_links(); ?>
                        </div>
                    <?php } 
                    blossom_floral_site_branding(); ?>
                    <div class="header-right">
                        <?php 
                        blossom_floral_search();
                        blossom_floral_header_cart(); ?>
                    </div>
                </div>
            </div>
            <div class="header-main">
                <div class="container">
                    <?php blossom_floral_primary_nagivation(); ?>
                </div>
            </div>
            <?php 
                blossom_floral_mobile_navigation();  
            ?>
            <?php 
        }else{ ?>
            <div class="header-top">
                <div class="container">
                    <div class="header-left">
                        <?php 
                        blossom_floral_secondary_navigation();
                        blossom_floral_social_links(); ?>
                    </div>
                    <div class="header-right">
                        <?php
                        blossom_floral_search();
                        blossom_floral_header_cart(); ?>
                    </div>
                </div>
            </div>
            <div class="header-middle">
                <div class="container">
                    <?php blossom_floral_site_branding(); ?>
                </div>
            </div>
            <div class="header-main">
                <div class="container">
                    <?php blossom_floral_primary_nagivation(); ?>
                </div>
            </div>
            <?php 
                blossom_floral_mobile_navigation();  
            ?>
        <?php } ?>
    </header>
    <?php
}

/**
 * Banner Section 
*/
function blossom_floral_banner(){
    $ed_banner      = get_theme_mod( 'ed_banner_section', 'slider_banner' );
    $slider_type    = get_theme_mod( 'slider_type', 'latest_posts' ); 
    $slider_cat     = get_theme_mod( 'slider_cat' );
    $posts_per_page = get_theme_mod( 'no_of_slides', 3 );    
    $ed_caption     = get_theme_mod( 'slider_caption', true );
    $slider_layout   = get_theme_mod( 'slider_layout', 'eight' );
    
    if( is_front_page() || is_home() ){ 
        
        if( $ed_banner == 'slider_banner' ){
            $args = array(
                'post_type'           => 'post',
                'post_status'         => 'publish',            
                'ignore_sticky_posts' => true,
            );
            
            if( $slider_type === 'cat' && $slider_cat ){
                $args['cat']            = $slider_cat; 
                $args['posts_per_page'] = -1;  
            }else{
                $args['posts_per_page'] = $posts_per_page;
            }
                
            $qry = new WP_Query( $args );
            
            if( $qry->have_posts() ){ ?>
                <div id="banner-section" class="site-banner slider-<?php echo esc_attr( $slider_layout ); ?>">
                    <div class="banner-wrapper owl-carousel">
                        <?php while( $qry->have_posts() ){ $qry->the_post(); ?>
                            <div class="item">
                                <?php 
                                echo '<div class="banner-img-wrap"><a href="' . esc_url( get_permalink( get_the_ID() ) ) . '">';
                                    if( has_post_thumbnail() ){
                                        the_post_thumbnail( 'blossom-floral-slider', array( 'itemprop' => 'image' ) );    
                                    }else{ 
                                        blossom_floral_get_fallback_svg( 'blossom-floral-slider' );//fallback
                                    }
                                echo '</a></div>';
                                if( $ed_caption ){ 
                                    if( $slider_layout == 'eight' ) echo '<div class="container">'; ?>
                                        <div class="banner-caption">
                                            <?php 
                                                echo '<div class="entry-meta">';
                                                    blossom_floral_category();
                                                echo '</div>';
                                                the_title( '<h2 class="banner-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h2>' );
                                            ?>
                                        </div>
                                    <?php
                                   if( $slider_layout == 'eight' ) echo '</div>';
                                } ?>
                            </div>
                        <?php } ?>                        
                    </div>
                </div>
                <?php
                wp_reset_postdata();
            }
        } 
    }    
}

function blossom_floral_post_thumbnail() {
    
    $image_size     = 'thumbnail'; 
    $ed_crop_blog   = get_theme_mod( 'ed_crop_blog', false );
    $sidebar        = blossom_floral_sidebar();
    $home_layout    = get_theme_mod( 'homepage_layout', 'two' );

    if( $home_layout == 'one'){
        $image_size = 'blossom-floral-blog-home';
    }else{
        $image_size = 'blossom-floral-blog-list-small';
    }
    
    if( is_home() || is_archive() || is_search() ){
       
        echo '<figure class="post-thumbnail"><a href="' . esc_url( get_permalink() ) . '" class="post-thumbnail">';
        if( has_post_thumbnail() ){
            if( $ed_crop_blog ){
                the_post_thumbnail( 'full', array( 'itemprop' => 'image' ) );
            }else{
                the_post_thumbnail( $image_size, array( 'itemprop' => 'image' ) );    
            }
        }else{
            blossom_floral_get_fallback_svg( $image_size );//fallback
        }
        echo '</a></figure>';
    }elseif( is_singular() ){
        $image_size = ( $sidebar ) ? 'blossom-floral-with-sidebar' : 'full';
        
        if( has_post_thumbnail() ){
            echo '<div class="post-thumbnail">';
            the_post_thumbnail( $image_size, array( 'itemprop' => 'image' ) );
            echo '</div>';    
        } 
    }
}


function blossom_floral_footer_bottom(){ ?>
    <div class="footer-bottom">
		<div class="container">
            <div class="footer-bottom__content-wrapper">
                <div class="site-info">            
                <?php
                    blossom_floral_get_footer_copyright();
                    echo esc_html__( 'Floral Fashion | Developed By ', 'floral-fashion' ); 
                    echo '<a href="' . esc_url( 'https://blossomthemes.com/' ) .'" rel="nofollow" target="_blank">' . esc_html__( 'Blossom Themes', 'floral-fashion' ) . '</a>.';                
                    printf( esc_html__( ' Powered by %s. ', 'floral-fashion' ), '<a href="'. esc_url( __( 'https://wordpress.org/', 'floral-fashion' ) ) .'" target="_blank">WordPress</a>' );
                    if( function_exists( 'the_privacy_policy_link' ) ){
                        the_privacy_policy_link();
                    }
                ?>               
                </div>
                <?php if( blossom_floral_social_links( false ) ) { ?>
                    <div class="footer-social-network">
                        <?php blossom_floral_social_links(); ?>
                    </div>
                    <?php
                } ?>
                <div class="footer-bottom-right">
                    <?php 
                        blossom_floral_footer_navigation();
                    ?>
                </div>
            </div>
		</div>
	</div>
    <?php
}

function blossom_floral_fonts_url(){
    $fonts_url = '';
    
    $primary_font       = get_theme_mod( 'primary_font', 'Inter' );
    $ig_primary_font    = blossom_floral_is_google_font( $primary_font );    
    $secondary_font     = get_theme_mod( 'secondary_font', 'Source Serif Pro' );
    $ig_secondary_font  = blossom_floral_is_google_font( $secondary_font );    
    $site_title_font    = get_theme_mod( 'site_title_font', array( 'font-family'=>'Crimson Pro', 'variant'=>'regular' ) );
    $ig_site_title_font = blossom_floral_is_google_font( $site_title_font['font-family'] );
        
    /* Translators: If there are characters in your language that are not
    * supported by respective fonts, translate this to 'off'. Do not translate
    * into your own language.
    */
    $primary    = _x( 'on', 'Primary Font: on or off', 'floral-fashion' );
    $secondary  = _x( 'on', 'Secondary Font: on or off', 'floral-fashion' );
    $site_title = _x( 'on', 'Site Title Font: on or off', 'floral-fashion' );
    
    
    if ( 'off' !== $primary || 'off' !== $secondary || 'off' !== $site_title ) {
        
        $font_families = array();
     
        if ( 'off' !== $primary && $ig_primary_font ) {
            $primary_variant = blossom_floral_check_varient( $primary_font, 'regular', true );
            if( $primary_variant ){
                $primary_var = ':' . $primary_variant;
            }else{
                $primary_var = '';    
            }            
            $font_families[] = $primary_font . $primary_var;
        }
         
        if ( 'off' !== $secondary && $ig_secondary_font ) {
            $secondary_variant = blossom_floral_check_varient( $secondary_font, 'regular', true );
            if( $secondary_variant ){
                $secondary_var = ':' . $secondary_variant;    
            }else{
                $secondary_var = '';
            }
            $font_families[] = $secondary_font . $secondary_var;
        }
        
        if ( 'off' !== $site_title && $ig_site_title_font ) {
            
            if( ! empty( $site_title_font['variant'] ) ){
                $site_title_var = ':' . blossom_floral_check_varient( $site_title_font['font-family'], $site_title_font['variant'] );    
            }else{
                $site_title_var = '';
            }
            $font_families[] = $site_title_font['font-family'] . $site_title_var;
        }
        
        $font_families = array_diff( array_unique( $font_families ), array('') );
        
        $query_args = array(
            'family' => urlencode( implode( '|', $font_families ) ),            
        );
        
        $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
    }

    if( get_theme_mod( 'ed_localgoogle_fonts', false ) ) {
        $fonts_url = blossom_floral_get_webfont_url( add_query_arg( $query_args, 'https://fonts.googleapis.com/css' ) );
    }
     
    return esc_url( $fonts_url );
}


function floral_fashion_dynamic_css(){
    
    $primary_font    = get_theme_mod( 'primary_font', 'Inter' );
    $primary_fonts   = blossom_floral_get_fonts( $primary_font, 'regular' );
    $secondary_font  = get_theme_mod( 'secondary_font', 'Source Serif Pro' );
    $secondary_fonts = blossom_floral_get_fonts( $secondary_font, 'regular' );
    $font_size       = get_theme_mod( 'font_size', 18 );
    
    $site_title_font      = get_theme_mod( 'site_title_font', array( 'font-family'=>'Crimson Pro', 'variant'=>'regular' ) );
    $site_title_fonts     = blossom_floral_get_fonts( $site_title_font['font-family'], $site_title_font['variant'] );
    $site_title_font_size = get_theme_mod( 'site_title_font_size', 30 );
    $logo_width           = get_theme_mod( 'logo_width', 150 );

	$primary_color    = get_theme_mod( 'primary_color', '#fdbea9' );
	$secondary_color  = get_theme_mod( 'secondary_color', '#ddbea9' );

	$shop_sec_image   = get_theme_mod( 'shop_bg_image' );
	$shop_sec_color   = get_theme_mod( 'shop_bg_color', '#F2CAB3' );
	$shop_type 		  = get_theme_mod( 'shop_bg', 'image' );

	$header_bg_img  = get_theme_mod( 'header_bg_image' );
	$header_bg_img2 = get_theme_mod( 'header_bg_image_two' );

	$newsletter_image = get_theme_mod( 'newsletter_bg_image' );

	$about_bg_image = get_theme_mod( 'about_bg_image' );

	$content_bg_img = get_theme_mod( 'content_bg_image' );
    $top_bg_img     = get_theme_mod( 'top_bg_image' );
    $left_bg_img    = get_theme_mod( 'left_bg_image' );
    $right_bg_img   = get_theme_mod( 'right_bg_image' );

	$footer_bg_img  = get_theme_mod( 'footer_bg_image' );
    $footer_top_img = get_theme_mod( 'footer_top_image' );
    $footer_btm_img = get_theme_mod( 'footer_bottom_image' );

    $rgb = blossom_floral_hex2rgb( blossom_floral_sanitize_hex_color( $primary_color ) );
	$rgb2 = blossom_floral_hex2rgb( blossom_floral_sanitize_hex_color( $secondary_color ) );

    echo "<style type='text/css' media='all'>"; ?>
    
	:root {
		--primary-color: <?php echo blossom_floral_sanitize_hex_color( $primary_color ); ?>;
		--primary-color-rgb: <?php printf('%1$s, %2$s, %3$s', $rgb[0], $rgb[1], $rgb[2] ); ?>;
		--secondary-color: <?php echo blossom_floral_sanitize_hex_color( $secondary_color ); ?>;
		--secondary-color-rgb: <?php printf('%1$s, %2$s, %3$s', $rgb2[0], $rgb2[1], $rgb2[2] ); ?>;
        --primary-font: <?php echo esc_html( $primary_fonts['font'] ); ?>;
        --secondary-font: <?php echo esc_html( $secondary_fonts['font'] ); ?>;
	}

    /*Typography*/

    .custom-logo-link img{
        width    : <?php echo absint( $logo_width ); ?>px;
        max-width: 100%;
	}
    
    .site-title{
        font-size   : <?php echo absint( $site_title_font_size ); ?>px;
        font-family : <?php echo esc_html( $site_title_fonts['font'] ); ?>;
        font-weight : <?php echo esc_html( $site_title_fonts['weight'] ); ?>;
        font-style  : <?php echo esc_html( $site_title_fonts['style'] ); ?>;
    }

	<?php if( $header_bg_img ){ ?>
        .site-header.style-one > .header-middle, .site-header.style-four > .header-middle {
            background-image: url( <?php echo esc_url( $header_bg_img ); ?> );
        }
    <?php } 
    
    if( $header_bg_img2 ){ ?>
        .site-header.header-img::before {
            content: "";
            background-image: url( <?php echo esc_url( $header_bg_img2 ); ?> );
        }
    <?php } 
    
    if( ( $shop_type == 'image' && $shop_sec_image ) || ( $shop_type == 'color' && $shop_sec_color ) ) { ?>  
        .product-section {
            <?php if( $shop_type == 'image' ){ ?>
                background-image: url( <?php echo esc_url( $shop_sec_image ); ?> );
            <?php }elseif( $shop_type == 'color' ){ ?>
                background-color: <?php echo blossom_floral_sanitize_hex_color( $shop_sec_color ); ?>;
            <?php } ?>
        }
    <?php }
    
    if( $about_bg_image ){ ?>
        .about-section{
            background-image: url(<?php echo esc_url( $about_bg_image ); ?>);
        }
    <?php }

    if( $content_bg_img ){ ?>
        .blog .site-content {
            background-image: url(<?php echo esc_url( $content_bg_img ); ?>);
        }
    <?php }

    if( $top_bg_img ){ ?>
        .blog .site-content .page-grid::before {
            background-image: url(<?php echo esc_url( $top_bg_img ); ?>);
        }
    <?php }

    if( $left_bg_img ){ ?>
        .blog .site-content::before {
            background-image: url(<?php echo esc_url( $left_bg_img ); ?>);
        }
    <?php }

    if( $right_bg_img ){ ?>
        .blog .site-content::after {
            background-image: url(<?php echo esc_url( $right_bg_img ); ?>);
        }
    <?php }

    if( $newsletter_image ){ ?>
        .newsletter-section {
            background-image: url( <?php echo esc_url( $newsletter_image ); ?> );
        }
    <?php }

    if( $footer_bg_img ){ ?>
        .site-footer {
            background-image: url( <?php echo esc_url( $footer_bg_img ); ?> );
        }
    <?php }

    if( $footer_top_img ){ ?>
        .site-footer .footer-mid::before {
            background-image: url( <?php echo esc_url( $footer_top_img ); ?> );
        }
    <?php } 

    if( $footer_btm_img ){ ?>
        .site-footer .footer-bottom .footer-bottom__content-wrapper::before {
            background-image: url( <?php echo esc_url( $footer_btm_img ); ?> );
        }
    <?php } ?>

    .post .entry-footer .btn-link::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='14.19' height='14.27' viewBox='0 0 14.19 14.27'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(-216.737 -1581.109)'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(217.445 1594.672) rotate(-45)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,0,5.1,5.1,0,10.193' transform='translate(219.262 1585.567) rotate(-45)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E");
    }

    .navigation.pagination .nav-links .next:hover:after {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='13.821' height='11.608' viewBox='0 0 13.821 11.608'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(974.347 -1275.499) rotate(45)' opacity='0.9'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(217.445 1594.672) rotate(-45)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,0,5.1,5.1,0,10.193' transform='translate(219.262 1585.567) rotate(-45)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E");
    }

    nav.post-navigation .meta-nav::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='14.19' height='14.27' viewBox='0 0 14.19 14.27'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(230.936 0.854)' opacity='0.9'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(-217.453 12.709) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,10.194,5.1,5.1,0,0' transform='translate(-226.479 10.812) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E ");
    }
       
    nav.post-navigation .meta-nav:hover::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='14.19' height='14.27' viewBox='0 0 14.19 14.27'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(230.936 0.854)' opacity='0.9'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(-217.453 12.709) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,10.194,5.1,5.1,0,0' transform='translate(-226.479 10.812) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E ");
    }
    
    .navigation.pagination .nav-links .prev:hover::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='21.956' height='13.496' viewBox='0 0 21.956 13.496'%3E%3Cg id='Group_1417' data-name='Group 1417' transform='translate(742.952 0.612)'%3E%3Cpath id='Path_1' data-name='Path 1' d='M1083.171,244.108h-20.837' transform='translate(-1804.667 -237.962)' fill='none' stroke='%2388b4cf' stroke-linecap='round' stroke-width='1'%3E%3C/path%3E%3Cpath id='Path_2' data-name='Path 2' d='M1093.614,226.065c-.695,2.593-1.669,4.985-6.7,6.143' transform='translate(-1829.267 -226.065)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'%3E%3C/path%3E%3Cpath id='Path_3' data-name='Path 3' d='M1093.614,232.208c-.695-2.593-1.669-4.985-6.7-6.143' transform='translate(-1829.267 -219.937)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'%3E%3C/path%3E%3C/g%3E%3C/svg%3E");
    }

    blockquote::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='127.432' height='110.665' viewBox='0 0 127.432 110.665'%3E%3Cg id='Group_1443' data-name='Group 1443' transform='translate(0 0)' opacity='0.3'%3E%3Cpath id='Path_5841' data-name='Path 5841' d='M194.147,345.773c-3.28,2.743-6.38,5.4-9.538,7.955-2.133,1.724-4.343,3.3-6.522,4.934-6.576,4.932-13.3,5.586-20.243,1.173-2.939-1.868-4.314-5.268-5.477-8.714a68.381,68.381,0,0,1-2.375-9.783c-.994-5.555-2.209-11.138-1.557-16.906.577-5.112,1.16-10.251,2.163-15.248a23.117,23.117,0,0,1,3.01-7.026c2.8-4.7,5.735-9.276,8.779-13.732a23.928,23.928,0,0,1,4.793-5.371c2.207-1.72,3.608-4.17,5.148-6.6,3.216-5.068,6.556-10.013,9.8-15.052a28.681,28.681,0,0,0,1.475-3.084c.163-.338.31-.795.563-.943,2.775-1.632,5.518-3.377,8.376-4.752,2.016-.97,3.528,1.238,5.25,2.057a3.4,3.4,0,0,1-.148,1.769c-1.535,3.621-3.138,7.2-4.71,10.8-3.534,8.085-7.357,16-10.514,24.308-3.248,8.542-6.275,17.324-6.5,27.026-.065,2.869.266,5.75.374,8.627.065,1.753,1.017,1.914,2.044,1.753a11.21,11.21,0,0,0,7.146-4.324c1.41-1.752,2.246-1.821,3.817-.239,2.013,2.029,3.923,4.218,5.856,6.367a1.677,1.677,0,0,1,.429,1.023c-.151,3.187-.352,6.379-2.323,8.826C191.077,343.331,191.107,343.7,194.147,345.773Z' transform='translate(-70.424 -252.194)' fill='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>'/%3E%3Cpath id='Path_5842' data-name='Path 5842' d='M259.193,344.341c-4.6,5.231-8.984,10.521-15.185,12.561a11.207,11.207,0,0,0-3.233,2.286c-5.3,4.46-11.216,4.268-17.085,2.977-4.218-.928-6.7-5.277-7.252-10.588-.948-9.07.893-17.566,3.187-26,.1-.381.287-.73.373-1.114,1.88-8.435,5.937-15.587,9.2-23.164,2.257-5.249,5.674-9.732,8.694-14.758.6,1.231.936,2.1,1.4,2.854.947,1.552,2.144,1.065,2.942-.529a12.559,12.559,0,0,0,.69-2.028c.39-1.313,1.017-1.885,2.24-.981-.207-2.706-.034-5.343,2.121-6.4.81-.4,2.093.691,3.288,1.15.659-1.414,1.61-3.271,2.38-5.236a4.422,4.422,0,0,0-.234-2.1c-.3-1.353-.733-2.666-.974-4.032a11.511,11.511,0,0,1,1.917-8.21c1.1-1.825,2.033-3.8,3.059-5.687,2.014-3.709,4.517-4.035,7.155-.948a17.668,17.668,0,0,0,2.386,2.7,5.03,5.03,0,0,0,2.526.767,7.3,7.3,0,0,0,2.09-.458c-.477,1.277-.81,2.261-1.2,3.2-4.945,11.79-10.1,23.454-14.784,35.4-3.468,8.844-6.331,18.054-9.458,27.1a6.573,6.573,0,0,0-.226.964c-.649,3.651.393,4.769,3.4,4.056,2.592-.618,4.313-3.327,6.743-4.071a16.177,16.177,0,0,1,5.847-.563c1.236.087,2.6,3.97,2.248,6.047-.7,4.12-1.9,8.009-4.311,11.09C258.068,341.977,257.566,343.062,259.193,344.341Z' transform='translate(-216.183 -252.301)' fill='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>'/%3E%3C/g%3E%3C/svg%3E%0A");
    }

    .comments-area .comment-list .comment .comment-body .reply .comment-reply-link::before,
    .comments-area ol .comment .comment-body .reply .comment-reply-link::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='14.19' height='14.27' viewBox='0 0 14.19 14.27'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(230.936 0.854)' opacity='0.9'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(-217.453 12.709) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,10.194,5.1,5.1,0,0' transform='translate(-226.479 10.812) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E ");
    }

    .comments-area .comment-list .comment .comment-body .reply .comment-reply-link:hover::before,
    .comments-area ol .comment .comment-body .reply .comment-reply-link:hover::before {
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='14.19' height='14.27' viewBox='0 0 14.19 14.27'%3E%3Cg id='Group_5482' data-name='Group 5482' transform='translate(230.936 0.854)' opacity='0.9'%3E%3Cpath id='Path_26475' data-name='Path 26475' d='M0,0H12.821' transform='translate(-217.453 12.709) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>' stroke-linecap='round' stroke-width='1'/%3E%3Cpath id='Path_26476' data-name='Path 26476' d='M0,10.194,5.1,5.1,0,0' transform='translate(-226.479 10.812) rotate(-135)' fill='none' stroke='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ); ?>' stroke-linecap='round' stroke-linejoin='round' stroke-width='1'/%3E%3C/g%3E%3C/svg%3E ");
    }

    select{
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='15' height='8' viewBox='0 0 15 8'%3E%3Cpath id='Polygon_25' data-name='Polygon 25' d='M7.5,0,15,8H0Z' transform='translate(15 8) rotate(180)' fill='<?php echo blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $secondary_color ) ); ?>'/%3E%3C/svg%3E%0A");
    }  
           
    <?php echo "</style>";
}
add_action( 'wp_head', 'floral_fashion_dynamic_css', 99 );

function blossom_floral_gutenberg_inline_style(){

	$primary_font    = get_theme_mod( 'primary_font', 'Inter' );
	$primary_fonts   = blossom_floral_get_fonts( $primary_font, 'regular' );
	$secondary_font  = get_theme_mod( 'secondary_font', 'Source Serif Pro' );
	$secondary_fonts = blossom_floral_get_fonts( $secondary_font, 'regular' );
	$primary_color    = get_theme_mod( 'primary_color', '#fdbea9' );
	$secondary_color  = get_theme_mod( 'secondary_color', '#ddbea9' );

	$rgb  = blossom_floral_hex2rgb( blossom_floral_sanitize_hex_color( $primary_color ) );
	$rgb2 = blossom_floral_hex2rgb( blossom_floral_sanitize_hex_color( $secondary_color ) );
	
	$custom_css = ':root .block-editor-page {
		--primary-font: ' . esc_html($primary_fonts['font']) . ';
		--secondary-font: ' . esc_html($secondary_fonts['font']) . ';
		--primary-color: ' . blossom_floral_sanitize_hex_color($primary_color) . ';
		--primary-color-rgb: ' . sprintf('%1$s, %2$s, %3$s', $rgb[0], $rgb[1], $rgb[2]) . ';
		--secondary-color: ' . blossom_floral_sanitize_hex_color($secondary_color) . ';
		--secondary-color-rgb: ' . sprintf('%1$s, %2$s, %3$s', $rgb2[0], $rgb2[1], $rgb2[2]) . ';
	}

	blockquote.wp-block-quote::before {
		background-image: url(\'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="127.432" height="110.665" viewBox="0 0 127.432 110.665"%3E%3Cg id="Group_1443" data-name="Group 1443" transform="translate(0 0)" opacity="0.3"%3E%3Cpath id="Path_5841" data-name="Path 5841" d="M194.147,345.773c-3.28,2.743-6.38,5.4-9.538,7.955-2.133,1.724-4.343,3.3-6.522,4.934-6.576,4.932-13.3,5.586-20.243,1.173-2.939-1.868-4.314-5.268-5.477-8.714a68.381,68.381,0,0,1-2.375-9.783c-.994-5.555-2.209-11.138-1.557-16.906.577-5.112,1.16-10.251,2.163-15.248a23.117,23.117,0,0,1,3.01-7.026c2.8-4.7,5.735-9.276,8.779-13.732a23.928,23.928,0,0,1,4.793-5.371c2.207-1.72,3.608-4.17,5.148-6.6,3.216-5.068,6.556-10.013,9.8-15.052a28.681,28.681,0,0,0,1.475-3.084c.163-.338.31-.795.563-.943,2.775-1.632,5.518-3.377,8.376-4.752,2.016-.97,3.528,1.238,5.25,2.057a3.4,3.4,0,0,1-.148,1.769c-1.535,3.621-3.138,7.2-4.71,10.8-3.534,8.085-7.357,16-10.514,24.308-3.248,8.542-6.275,17.324-6.5,27.026-.065,2.869.266,5.75.374,8.627.065,1.753,1.017,1.914,2.044,1.753a11.21,11.21,0,0,0,7.146-4.324c1.41-1.752,2.246-1.821,3.817-.239,2.013,2.029,3.923,4.218,5.856,6.367a1.677,1.677,0,0,1,.429,1.023c-.151,3.187-.352,6.379-2.323,8.826C191.077,343.331,191.107,343.7,194.147,345.773Z" transform="translate(-70.424 -252.194)" fill="' . blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ) . '"/%3E%3Cpath id="Path_5842" data-name="Path 5842" d="M259.193,344.341c-4.6,5.231-8.984,10.521-15.185,12.561a11.207,11.207,0,0,0-3.233,2.286c-5.3,4.46-11.216,4.268-17.085,2.977-4.218-.928-6.7-5.277-7.252-10.588-.948-9.07.893-17.566,3.187-26,.1-.381.287-.73.373-1.114,1.88-8.435,5.937-15.587,9.2-23.164,2.257-5.249,5.674-9.732,8.694-14.758.6,1.231.936,2.1,1.4,2.854.947,1.552,2.144,1.065,2.942-.529a12.559,12.559,0,0,0,.69-2.028c.39-1.313,1.017-1.885,2.24-.981-.207-2.706-.034-5.343,2.121-6.4.81-.4,2.093.691,3.288,1.15.659-1.414,1.61-3.271,2.38-5.236a4.422,4.422,0,0,0-.234-2.1c-.3-1.353-.733-2.666-.974-4.032a11.511,11.511,0,0,1,1.917-8.21c1.1-1.825,2.033-3.8,3.059-5.687,2.014-3.709,4.517-4.035,7.155-.948a17.668,17.668,0,0,0,2.386,2.7,5.03,5.03,0,0,0,2.526.767,7.3,7.3,0,0,0,2.09-.458c-.477,1.277-.81,2.261-1.2,3.2-4.945,11.79-10.1,23.454-14.784,35.4-3.468,8.844-6.331,18.054-9.458,27.1a6.573,6.573,0,0,0-.226.964c-.649,3.651.393,4.769,3.4,4.056,2.592-.618,4.313-3.327,6.743-4.071a16.177,16.177,0,0,1,5.847-.563c1.236.087,2.6,3.97,2.248,6.047-.7,4.12-1.9,8.009-4.311,11.09C258.068,341.977,257.566,343.062,259.193,344.341Z" transform="translate(-216.183 -252.301)" fill="' . blossom_floral_hash_to_percent23( blossom_floral_sanitize_hex_color( $primary_color ) ) . '"/%3E%3C/g%3E%3C/svg%3E%0A\');
	}';

	return $custom_css;
}