<?php
/**
 * Typography control class.
 *
 * @since  1.0.0
 * @access public
 */

class Keto_Organic_Diet_Control_Typography extends WP_Customize_Control {

	/**
	 * The type of customize control being rendered.
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $type = 'keto-organic-diet-typography';

	/**
	 * Array 
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $l10n = array();

	/**
	 * Set up our control.
	 *
	 * @since  1.0.0
	 * @access public
	 * @param  object  $manager
	 * @param  string  $id
	 * @param  array   $args
	 * @return void
	 */
	public function __construct( $manager, $id, $args = array() ) {

		// Let the parent class do its thing.
		parent::__construct( $manager, $id, $args );

		// Make sure we have labels.
		$this->l10n = wp_parse_args(
			$this->l10n,
			array(
				'color'       => esc_html__( 'Font Color', 'keto-organic-diet' ),
				'family'      => esc_html__( 'Font Family', 'keto-organic-diet' ),
				'size'        => esc_html__( 'Font Size',   'keto-organic-diet' ),
				'weight'      => esc_html__( 'Font Weight', 'keto-organic-diet' ),
				'style'       => esc_html__( 'Font Style',  'keto-organic-diet' ),
				'line_height' => esc_html__( 'Line Height', 'keto-organic-diet' ),
				'letter_spacing' => esc_html__( 'Letter Spacing', 'keto-organic-diet' ),
			)
		);
	}

	/**
	 * Enqueue scripts/styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function enqueue() {
		wp_enqueue_script( 'keto-organic-diet-ctypo-customize-controls' );
		wp_enqueue_style(  'keto-organic-diet-ctypo-customize-controls' );
	}

	/**
	 * Add custom parameters to pass to the JS via JSON.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function to_json() {
		parent::to_json();

		// Loop through each of the settings and set up the data for it.
		foreach ( $this->settings as $setting_key => $setting_id ) {

			$this->json[ $setting_key ] = array(
				'link'  => $this->get_link( $setting_key ),
				'value' => $this->value( $setting_key ),
				'label' => isset( $this->l10n[ $setting_key ] ) ? $this->l10n[ $setting_key ] : ''
			);

			if ( 'family' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_families();

			elseif ( 'weight' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_weight_choices();

			elseif ( 'style' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_style_choices();
		}
	}

	/**
	 * Underscore JS template to handle the control's output.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function content_template() { ?>

		<# if ( data.label ) { #>
			<span class="customize-control-title">{{ data.label }}</span>
		<# } #>

		<# if ( data.description ) { #>
			<span class="description customize-control-description">{{{ data.description }}}</span>
		<# } #>

		<ul>

		<# if ( data.family && data.family.choices ) { #>

			<li class="typography-font-family">

				<# if ( data.family.label ) { #>
					<span class="customize-control-title">{{ data.family.label }}</span>
				<# } #>

				<select {{{ data.family.link }}}>

					<# _.each( data.family.choices, function( label, choice ) { #>
						<option value="{{ choice }}" <# if ( choice === data.family.value ) { #> selected="selected" <# } #>>{{ label }}</option>
					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.weight && data.weight.choices ) { #>

			<li class="typography-font-weight">

				<# if ( data.weight.label ) { #>
					<span class="customize-control-title">{{ data.weight.label }}</span>
				<# } #>

				<select {{{ data.weight.link }}}>

					<# _.each( data.weight.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.weight.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.style && data.style.choices ) { #>

			<li class="typography-font-style">

				<# if ( data.style.label ) { #>
					<span class="customize-control-title">{{ data.style.label }}</span>
				<# } #>

				<select {{{ data.style.link }}}>

					<# _.each( data.style.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.style.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.size ) { #>

			<li class="typography-font-size">

				<# if ( data.size.label ) { #>
					<span class="customize-control-title">{{ data.size.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.size.link }}} value="{{ data.size.value }}" />

			</li>
		<# } #>

		<# if ( data.line_height ) { #>

			<li class="typography-line-height">

				<# if ( data.line_height.label ) { #>
					<span class="customize-control-title">{{ data.line_height.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.line_height.link }}} value="{{ data.line_height.value }}" />

			</li>
		<# } #>

		<# if ( data.letter_spacing ) { #>

			<li class="typography-letter-spacing">

				<# if ( data.letter_spacing.label ) { #>
					<span class="customize-control-title">{{ data.letter_spacing.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.letter_spacing.link }}} value="{{ data.letter_spacing.value }}" />

			</li>
		<# } #>

		</ul>
	<?php }

	/**
	 * Returns the available fonts.  Fonts should have available weights, styles, and subsets.
	 *
	 * @todo Integrate with Google fonts.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_fonts() { return array(); }

	/**
	 * Returns the available font families.
	 *
	 * @todo Pull families from `get_fonts()`.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	function get_font_families() {

		return array(
			'' => __( 'No Fonts', 'keto-organic-diet' ),
        'Abril Fatface' => __( 'Abril Fatface', 'keto-organic-diet' ),
        'Acme' => __( 'Acme', 'keto-organic-diet' ),
        'Anton' => __( 'Anton', 'keto-organic-diet' ),
        'Architects Daughter' => __( 'Architects Daughter', 'keto-organic-diet' ),
        'Arimo' => __( 'Arimo', 'keto-organic-diet' ),
        'Arsenal' => __( 'Arsenal', 'keto-organic-diet' ),
        'Arvo' => __( 'Arvo', 'keto-organic-diet' ),
        'Alegreya' => __( 'Alegreya', 'keto-organic-diet' ),
        'Alfa Slab One' => __( 'Alfa Slab One', 'keto-organic-diet' ),
        'Averia Serif Libre' => __( 'Averia Serif Libre', 'keto-organic-diet' ),
        'Albert Sans' => __( 'Albert Sans', 'keto-organic-diet' ),
        'Bangers' => __( 'Bangers', 'keto-organic-diet' ),
        'Boogaloo' => __( 'Boogaloo', 'keto-organic-diet' ),
        'Bad Script' => __( 'Bad Script', 'keto-organic-diet' ),
        'Bitter' => __( 'Bitter', 'keto-organic-diet' ),
        'Bree Serif' => __( 'Bree Serif', 'keto-organic-diet' ),
        'BenchNine' => __( 'BenchNine', 'keto-organic-diet' ),
        'Cabin' => __( 'Cabin', 'keto-organic-diet' ),
        'Cardo' => __( 'Cardo', 'keto-organic-diet' ),
        'Courgette' => __( 'Courgette', 'keto-organic-diet' ),
        'Cherry Swash' => __( 'Cherry Swash', 'keto-organic-diet' ),
        'Cormorant Garamond' => __( 'Cormorant Garamond', 'keto-organic-diet' ),
        'Crimson Text' => __( 'Crimson Text', 'keto-organic-diet' ),
        'Cuprum' => __( 'Cuprum', 'keto-organic-diet' ),
        'Cookie' => __( 'Cookie', 'keto-organic-diet' ),
        'Chewy' => __( 'Chewy', 'keto-organic-diet' )
		);
	}

	/**
	 * Returns the available font weights.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_weight_choices() {

		return array(
			'' => esc_html__( 'No Fonts weight', 'keto-organic-diet' ),
			'100' => esc_html__( 'Thin',       'keto-organic-diet' ),
			'300' => esc_html__( 'Light',      'keto-organic-diet' ),
			'400' => esc_html__( 'Normal',     'keto-organic-diet' ),
			'500' => esc_html__( 'Medium',     'keto-organic-diet' ),
			'700' => esc_html__( 'Bold',       'keto-organic-diet' ),
			'900' => esc_html__( 'Ultra Bold', 'keto-organic-diet' ),
		);
	}

	/**
	 * Returns the available font styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_style_choices() {

		return array(
			'' => esc_html__( 'No Fonts Style', 'keto-organic-diet' ),
			'normal'  => esc_html__( 'Normal', 'keto-organic-diet' ),
			'italic'  => esc_html__( 'Italic', 'keto-organic-diet' ),
			'oblique' => esc_html__( 'Oblique', 'keto-organic-diet' )
		);
	}
}
